"""
Serial Data Capture Script
Reads ASCII measurements from a serial port and records them to a CSV file's Z column.
"""

import serial
import csv
import time
from datetime import datetime
import argparse
import sys


def capture_serial_data(port, baudrate, output_file, duration=None, sample_count=None):
    """
    Capture data from serial port and write to CSV.
    
    Args:
        port: Serial port name (e.g., 'COM3' on Windows, '/dev/ttyUSB0' on Linux)
        output_file: Path to output CSV file
        duration: Optional duration in seconds to capture data
        sample_count: Optional number of samples to capture
    """
    
    print(f"Opening serial port {port} at {baudrate} baud...")
    
    try:
        # Open serial port
        ser = serial.Serial(
            port=port,
            baudrate=baudrate,
            bytesize=serial.EIGHTBITS,
            parity=serial.PARITY_NONE,
            stopbits=serial.STOPBITS_ONE,
            timeout=1.0
        )
        
        # Wait for connection to stabilize
        time.sleep(2)
        
        # Clear any buffered data
        ser.reset_input_buffer()
        
        print(f"Serial port opened successfully. Writing to {output_file}")
        print("Press Ctrl+C to stop capture...\n")
        
        # Open CSV file for writing
        with open(output_file, 'w', newline='') as csvfile:
            writer = csv.writer(csvfile)
            
            # Write header
            writer.writerow(['Timestamp', 'Index', 'Z'])
            
            start_time = time.time()
            sample_index = 0
            
            try:
                while True:
                    # Check stopping conditions
                    if duration and (time.time() - start_time) >= duration:
                        print(f"\nReached duration limit of {duration} seconds")
                        break
                    
                    if sample_count and sample_index >= sample_count:
                        print(f"\nReached sample count limit of {sample_count}")
                        break
                    
                    # Read line from serial port
                    if ser.in_waiting > 0:
                        try:
                            line = ser.readline().decode('ascii').strip()
                            
                            if line:
                                # Try to convert to float
                                try:
                                    z_value = float(line)
                                    timestamp = datetime.now().isoformat()
                                    
                                    # Write to CSV
                                    writer.writerow([timestamp, sample_index, z_value])
                                    
                                    # Print progress
                                    if sample_index % 100 == 0:
                                        print(f"Samples captured: {sample_index} | Latest value: {z_value:.4f}")
                                    
                                    sample_index += 1
                                    
                                except ValueError:
                                    print(f"Warning: Could not convert '{line}' to float, skipping...")
                        
                        except UnicodeDecodeError:
                            print("Warning: Could not decode data as ASCII, skipping...")
            
            except KeyboardInterrupt:
                print("\n\nCapture stopped by user")
        
        print(f"\nCapture complete! Total samples: {sample_index}")
        print(f"Data saved to: {output_file}")
        
    except serial.SerialException as e:
        print(f"Error opening serial port: {e}", file=sys.stderr)
        sys.exit(1)
    
    except Exception as e:
        print(f"Unexpected error: {e}", file=sys.stderr)
        sys.exit(1)
    
    finally:
        if 'ser' in locals() and ser.is_open:
            ser.close()
            print("Serial port closed")


def list_serial_ports():
    """List available serial ports"""
    import serial.tools.list_ports
    
    ports = serial.tools.list_ports.comports()
    
    if not ports:
        print("No serial ports found")
        return
    
    print("Available serial ports:")
    for port in ports:
        print(f"  {port.device} - {port.description}")


def main():
    parser = argparse.ArgumentParser(
        description='Capture ASCII data from serial port and save to CSV',
        formatter_class=argparse.RawDescriptionHelpFormatter,
        epilog="""
Examples:
  # Capture from COM3
  python serial_capture.py COM3 data.csv
  
  # Capture for 60 seconds
  python serial_capture.py COM3 data.csv --duration 60
  
  # Capture 1000 samples
  python serial_capture.py COM3 data.csv --samples 1000
  
  # List available ports
  python serial_capture.py --list
        """
    )
    
    parser.add_argument('--list', action='store_true',
                        help='List available serial ports and exit')
    
    parser.add_argument('port', nargs='?',
                        help='Serial port (e.g., COM3, /dev/ttyUSB0)')
       
    parser.add_argument('output', nargs='?', default='serial_data.csv',
                        help='Output CSV file (default: serial_data.csv)')
    
    parser.add_argument('-d', '--duration', type=float,
                        help='Capture duration in seconds')
    
    parser.add_argument('-s', '--samples', type=int,
                        help='Number of samples to capture')
    
    args = parser.parse_args()
    
    # Handle --list flag
    if args.list:
        list_serial_ports()
        return
    
    # Validate required arguments
    if not args.port:
        parser.error("port is required (or use --list to see available ports)")
    
    # Run capture
    capture_serial_data(
        port=args.port,
        baudrate=115200,
        output_file=args.output,
        duration=args.duration,
        sample_count=args.samples
    )


if __name__ == '__main__':
    main()
