import plotly.graph_objects as go
import json
import os

def create_floor_plan_editor():
    """Interactive tool to create apartment floor plan"""
    
    # Initial empty floor plan
    fig = go.Figure()
    
    # Add grid
    fig.update_xaxes(
        range=[0, 5],
        showgrid=True,
        gridwidth=1,
        gridcolor='lightgray',
        dtick=1
    )
    fig.update_yaxes(
        range=[0, 10],
        showgrid=True,
        gridwidth=1,
        gridcolor='lightgray',
        dtick=1
    )
    
    fig.update_layout(
        title='Floor Plan Editor - Click to add walls, Save locations in code',
        xaxis_title='X Position (m)',
        yaxis_title='Y Position (m)',
        width=1000,
        height=800,
        plot_bgcolor='white',
        hovermode='closest'
    )
    
    # Instructions
    print("\n=== FLOOR PLAN EDITOR ===")
    print("1. Note the coordinates where you want to place walls")
    print("2. Update the 'walls' list below with wall coordinates")
    print("3. Run this script to visualize your floor plan")
    print("4. Copy the walls list to plot2.py\n")
    
    return fig

def draw_floor_plan(walls=None, rooms=None):
    """
    Draw floor plan with walls and room labels
    
    Parameters:
    - walls: List of wall dictionaries with 'x0', 'y0', 'x1', 'y1'
    - rooms: List of room dictionaries with 'name', 'x', 'y'
    """
    if walls is None:
        # Example floor plan - modify these coordinates for your apartment
        walls = [
            # Outer walls
            {'x0': 0, 'y0': 0, 'x1': 3.2+1.2+5.0, 'y1': 0},      # Bottom
            {'x0': 3.2+1.2+5.0, 'y0': 0, 'x1': 3.2+1.2+5.0, 'y1': 2.42+3.50-1.35},    # Right
            {'x0': 3.2+1.2+5.0, 'y0': 2.42+3.50-1.35, 'x1': 0, 'y1': 2.42+3.50-1.35},    # Top
            {'x0': 0, 'y0': 2.42+3.50-1.35, 'x1': 0, 'y1': 0},      # Left 
            
            # Interior walls - Living room/Kitchen divider
            {'x0': 3.2+1.2, 'y0': 1, 'x1': 3.2+1.2+5, 'y1': 1},
            
            # Bedroom walls
            {'x0': 0, 'y0': 3.5-1.35, 'x1': 3.2, 'y1': 3.5-1.35},
            {'x0': 3.2, 'y0': 2.42+3.50-1.35, 'x1': 3.2, 'y1': 2.42+3.50-1.35-2.42},
            {'x0': 3.2, 'y0': 0, 'x1': 3.2, 'y1': 3.50-1.35},
            
            # Bathroom wall
            {'x0': 3.2+1.2, 'y0': 0, 'x1': 3.2+1.2, 'y1': 2.42+3.50-1.35},
            {'x0': 3.2, 'y0': 3.50-1.35+1, 'x1': 3.2+1.2, 'y1': 3.50-1.35+1},

            # Toilet wall
            {'x0': 3.2+1.2+5-1.85, 'y0': 0, 'x1': 3.2+1.2+5-1.85, 'y1': 1},
            {'x0': 3.2+1.2+5-1.85-.94, 'y0': 0, 'x1': 3.2+1.2+5-1.85-.94, 'y1': 1},

            # Closet wall
            {'x0': 3.2+1.2+.5, 'y0': 0, 'x1': 3.2+1.2+.5, 'y1': 1},
        ]
    
    if rooms is None:
        rooms = [
            {'name': 'Kitchen', 'x': 7, 'y': 3},
            {'name': 'Bedroom 1', 'x': 1, 'y': 3.5},
            {'name': 'Bedroom 2', 'x': 1, 'y': 1},
            {'name': 'Shower', 'x': 3.75, 'y': 4},
            {'name': 'Toilet', 'x': 7, 'y': .5},
            {'name': 'Entry', 'x': 8.5, 'y': .5},
            {'name': 'Closet', 'x': 4.5, 'y': .5},
        ]
    
    fig = go.Figure()
    
    # Draw walls
    for wall in walls:
        fig.add_shape(
            type="line",
            x0=wall['x0'], y0=wall['y0'],
            x1=wall['x1'], y1=wall['y1'],
            line=dict(color="black", width=3)
        )
    
    # Add room labels
    for room in rooms:
        fig.add_annotation(
            x=room['x'],
            y=room['y'],
            text=room['name'],
            showarrow=False,
            font=dict(size=12, color='gray'),
            opacity=0.5
        )
    
    # Add grid
    fig.update_xaxes(
        range=[0, 10],
        showgrid=True,
        gridwidth=1,
        gridcolor='lightgray',
        dtick=1
    )
    fig.update_yaxes(
        range=[0, 5],
        showgrid=True,
        gridwidth=1,
        gridcolor='lightgray',
        dtick=1
    )
    
    fig.update_layout(
        title='Apartment Floor Plan',
        xaxis_title='X Position (m)',
        yaxis_title='Y Position (m)',
        width=1000,
        height=500,
        plot_bgcolor='white',
        showlegend=False
    )
    
    return fig, walls, rooms

def save_floor_plan(walls, rooms, filename='floor_plan.json'):
    """Save floor plan to JSON file"""
    os.makedirs('plots', exist_ok=True)
    floor_plan = {
        'walls': walls,
        'rooms': rooms
    }
    with open(f'plots/{filename}', 'w') as f:
        json.dump(floor_plan, f, indent=2)
    print(f"\nFloor plan saved to plots/{filename}")

def load_floor_plan(filename='floor_plan.json'):
    """Load floor plan from JSON file"""
    try:
        with open(f'plots/{filename}', 'r') as f:
            floor_plan = json.load(f)
        return floor_plan['walls'], floor_plan['rooms']
    except FileNotFoundError:
        print(f"Floor plan file not found: plots/{filename}")
        return None, None

# Example usage
if __name__ == "__main__":
    # Create and display floor plan with example layout
    fig, walls, rooms = draw_floor_plan()
    fig.show()
    
    # Save floor plan
    save_floor_plan(walls, rooms)
    
    print("\n=== CUSTOMIZE YOUR FLOOR PLAN ===")
    print("Edit the 'walls' and 'rooms' lists in this script:")
    print("\nWalls format:")
    print("  {'x0': start_x, 'y0': start_y, 'x1': end_x, 'y1': end_y}")
    print("\nRooms format:")
    print("  {'name': 'Room Name', 'x': center_x, 'y': center_y}")
    print("\nThen run this script again to visualize your changes.")
    print("\nCopy the printed code below to use in plot2.py:\n")
    
    print("# Add to plot2.py:")
    print(f"walls = {walls}")
    print(f"\nrooms = {rooms}")